<?php
declare(strict_types=1);
header('Content-Type: text/html; charset=utf-8');

$hash = $_GET['hash'] ?? '';
$hash = trim($hash);

if ($hash === '' || !preg_match('/^[A-Za-z0-9+=]+$/', $hash)) {
    http_response_code(400);
    exit('درخواست نامعتبر است.');
}

function getClientIp(): string {
    // ArvanCloud / Proxies
    $candidates = [
        'HTTP_AR_REAL_IP',          // اگر آروان این هدر را بدهد
        'HTTP_CF_CONNECTING_IP',    // بعضی‌ها مشابه Cloudflare
        'HTTP_X_REAL_IP',
        'HTTP_X_FORWARDED_FOR',
    ];

    foreach ($candidates as $k) {
        if (!empty($_SERVER[$k])) {
            $ip = $_SERVER[$k];

            // اگر لیست بود، اولین IP واقعی کاربر
            if (strpos($ip, ',') !== false) {
                $ip = trim(explode(',', $ip)[0]);
            } else {
                $ip = trim($ip);
            }

            // فقط IP معتبر قبول کن
            if (filter_var($ip, FILTER_VALIDATE_IP)) {
                return $ip;
            }
        }
    }

    return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
}

/* ====== DB CONFIG ====== */
$dbHost = "127.0.0.1";
$dbName = "maja01";
$dbUser = "root";
$dbPass = "";

try {
    $pdo = new PDO(
        "mysql:host={$dbHost};dbname={$dbName};charset=utf8mb4",
        $dbUser,
        $dbPass,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]
    );

    /* ====== IP (Real client IP best-effort) ====== */
    $ip = getClientIp();

    // اگر X-Forwarded-For چندتا IP داشت، اولی IP واقعی کاربره
    if (strpos($ip, ',') !== false) {
        $ip = trim(explode(',', $ip)[0]);
    }

    // تبدیل IP به باینری برای ذخیره سریع (IPv4/IPv6)
    $ipBin = @inet_pton($ip);
    if ($ipBin === false) {
        $ip = '0.0.0.0';
        $ipBin = inet_pton('0.0.0.0');
    }

    /* ====== Rate limit (IP-based) ====== */
    $maxFails  = 20;  // حداکثر تلاش ناموفق
    $windowMin = 10;  // بازه شمارش خطا (دقیقه)
    $blockMin  = 15;  // مدت بلاک (دقیقه)

    // تعداد خطاها و زمان آخرین خطا در windowMin اخیر
    $check = $pdo->prepare("
        SELECT MAX(created_at) AS last_fail, COUNT(*) AS fails
        FROM verify_attempts
        WHERE ip = ?
          AND ok = 0
          AND created_at >= (NOW() - INTERVAL ? MINUTE)
    ");
    $check->execute([$ipBin, $windowMin]);
    $row = $check->fetch() ?: ['fails' => 0, 'last_fail' => null];

    $fails = (int)$row['fails'];
    $lastFail = $row['last_fail'];

    if ($fails >= $maxFails && $lastFail) {
        // چند دقیقه از آخرین خطا گذشته؟
        $blockCheck = $pdo->prepare("SELECT TIMESTAMPDIFF(MINUTE, ?, NOW())");
        $blockCheck->execute([$lastFail]);
        $minsSinceLastFail = (int)$blockCheck->fetchColumn();

        if ($minsSinceLastFail < $blockMin) {
            http_response_code(429);
            $remain = $blockMin - $minsSinceLastFail;
            exit("تعداد درخواست‌ها زیاد است. لطفاً {$remain} دقیقه بعد دوباره تلاش کنید.");
        }
    }

    /* ====== Query member ====== */
    $stmt = $pdo->prepare("SELECT fn, ln, cm, status FROM members WHERE hash = ? LIMIT 1");
    $stmt->execute([$hash]);
    $m = $stmt->fetch();

    if (!$m) {
        // لاگ ناموفق
        $log = $pdo->prepare("
            INSERT INTO verify_attempts (ip, ip_text, hash, ok)
            VALUES (?, ?, ?, 0)
        ");
        $log->execute([$ipBin, $ip, $hash]);

        http_response_code(404);
        exit('کارت معتبر نمی‌باشد.');
    }

    // لاگ موفق
    $log = $pdo->prepare("
        INSERT INTO verify_attempts (ip, ip_text, hash, ok)
        VALUES (?, ?, ?, 1)
    ");
    $log->execute([$ipBin, $ip, $hash]);

    /* ====== Prepare output ====== */
    $fullName = htmlspecialchars($m['fn'] . ' ' . $m['ln'], ENT_QUOTES, 'UTF-8');

    $cm = (string)$m['cm'];
    $masked = str_repeat('*', max(0, strlen($cm) - 5)) . substr($cm, -5);

 
    $st = (int)$m['status'];

    $isValid = in_array($st, [0, 3], true); // فقط 0 و 3 معتبرند

    if ($st === 0) {
        $roleText = "عضو مجمع (در قید حیات)";
        $officialNoteText = "صاحب این کارت از  استانداران پیشین جمهوری اسلامی ایران می‌باشد. لطفاً با ایشان همکاری لازم را داشته باشید.";
    } elseif ($st === 3) {
        $roleText = "پرسنل معتبر";
        $officialNoteText = "صاحب این کارت از پرسنل مجمع استانداران جمهوری اسلامی ایران می‌باشد. لطفاً همکاری لازم را داشته باشید.";
    } else { // 1 یا 2
        $roleText = "عضو مجمع (فوت‌شده)";
        $officialNoteText = "این کارت متعلق به عضو فوت‌شده می‌باشد و فاقد اعتبار است. (صرفاً جهت سوابق)";
    }

    if ($isValid) {
        $statusText  = "این کارت تنها با عکس فوق و مشخصات زیر معتبر می باشد.";
        $statusColor = "#0f5132";
        $statusBg    = "#d1e7dd";
        $noteClass   = "official-note";
    } else {
        $statusText  = "کارت فاقد اعتبار است";
        $statusColor = "#842029";
        $statusBg    = "#f8d7da";
        $noteClass   = "official-note invalid";
    }

    // عکس بر اساس hash (اول png بعد jpg)
    $photoPath = "uploads/members/" . $hash . ".png";
    if (!file_exists(__DIR__ . "/" . $photoPath)) {
        $photoPath = "uploads/members/" . $hash . ".jpg";
    }
    if (!file_exists(__DIR__ . "/" . $photoPath)) {
        $photoPath = "uploads/members/default.jpg";
    }

} catch (Throwable $e) {
    http_response_code(500);
    exit('خطای سرور');
}
?>
<!doctype html>
<html lang="fa" dir="rtl">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>استعلام کارت عضویت</title>
<style>
:root{
  --green:#1c7c3a;
  --red:#b42318;
  --amber:#b58900;
  --ink:#1f2937;
  --muted:#6b7280;
  --paper:#ffffff;
}

*{box-sizing:border-box}
body{
  margin:0;
  font-family:tahoma,Arial;
  color:var(--ink);
  background:#eef1f5;
}

.page{
  min-height:100vh;
  display:flex;
  align-items:center;
  justify-content:center;
  padding:18px;
}

/* کارتِ اصلی */
.sheet{
  width:min(980px, 100%);
  background:var(--paper);
  border-radius:18px;
  overflow:hidden;
  box-shadow:0 18px 45px rgba(0,0,0,.10);
  border:1px solid rgba(0,0,0,.06);
  position:relative;
}

/* پس‌زمینه نقش‌دار شبیه کارت */
.sheet::before{
  content:"";
  position:absolute;
  inset:0;
  background-image:
    radial-gradient(circle at 20% 30%, rgba(0,0,0,.05) 0 2px, transparent 3px),
    radial-gradient(circle at 70% 60%, rgba(0,0,0,.05) 0 2px, transparent 3px);
  background-size:70px 70px;
  opacity:.6;
  pointer-events:none;
}

/* موج پایین شبیه کارت */
.sheet::after{
  content:"";
  position:absolute;
  left:0; right:0; bottom:0;
  height:120px;
  background:linear-gradient(180deg, rgba(0,0,0,0) 0%, rgba(0,0,0,.10) 35%, rgba(0,0,0,.18) 100%);
  pointer-events:none;
}

.header{
  position:relative;
  padding:26px 24px 10px;
  text-align:center;
}

.header .logo{
  width:64px;
  height:64px;
  object-fit:contain;
  margin-bottom:6px;
}

.header h1{
  margin:0;
  font-size:22px;
  color:var(--green);
  letter-spacing:-.2px;
}
.header p{
  margin:6px 0 0;
  font-size:13px;
  color:var(--muted);
}

.content{
  position:relative;
  padding:18px 24px 28px;
}

.grid{
  display:grid;
  grid-template-columns: 240px 1fr;
  gap:18px;
  align-items:center;
}

.photo{
  width:100%;
  aspect-ratio: 3/4;
  border-radius:50px;
  object-fit:cover;
  border:6px solid rgba(255,255,255,.9);
  box-shadow:0 10px 25px rgba(0,0,0,.12);
  background:#fff;
}

.info{
  background:rgba(255,255,255,.75);
  border:1px solid rgba(0,0,0,.06);
  border-radius:16px;
  padding:16px 16px;
  backdrop-filter: blur(2px);
}

.badge{
  display:inline-flex;
  align-items:center;
  gap:8px;
  padding:9px 14px;
  border-radius:999px;
  font-weight:700;
  font-size:14px;
  margin-bottom:12px;
}

.row{
  display:flex;
  gap:10px;
  padding:10px 0;
  border-bottom:1px dashed rgba(0,0,0,.10);
}
.row:last-child{border-bottom:none}
.k{
  min-width:140px;
  color:#374151;
  font-weight:700;
}
.v{
  font-weight:700;
  color:#111827;
}

.footer{
  position:relative;
  padding:18px 24px 22px;
  display:flex;
  flex-wrap:wrap;
  gap:10px;
  align-items:center;
  justify-content:space-between;
  color:#374151;
  font-size:12px;
}

.footer .note{
  max-width:680px;
  line-height:1.8;
  color:#374151;
  font-weight:700;
}
.footer .site{
  font-weight:800;
  letter-spacing:.3px;
}

/* ریسپانسیو */
@media (max-width:720px){
  .grid{grid-template-columns:1fr}
  .k{min-width:110px}
}

.official-note{
  margin-top:18px;
  padding:12px 14px;
  background:rgba(28,124,58,.08);
  border:1px solid rgba(28,124,58,.25);
  color:#1c7c3a;
  border-radius:12px;
  font-weight:700;
  font-size:14px;
  line-height:1.8;
}

.official-note.invalid{
  background:rgba(180,35,24,.08);
  border:1px solid rgba(180,35,24,.25);
  color:#b42318;
}
@media (max-width:720px){

  .grid{
    grid-template-columns:1fr !important;
    text-align:center;
  }

  .photo{
    width:70% !important;
    margin:0 auto 20px auto !important;
    border-radius:30px !important;
  }

  .info{
    text-align:right;
  }

}

</style>
</head>
<body>
  <div class="page">
    <div class="sheet">
      <div class="header">
         <img class="logo" src="uploads/logo.png" alt="logo">
        <h1>وزارت کشور</h1>
        <h1>مجمع استانداران جمهوری اسلامی ایران</h1>
        <h3>استعلام اصالت کارت شناسایی</h3>
      </div>

      <div class="content">
        <div class="grid">
          <img src="<?= htmlspecialchars($photoPath) ?>" class="photo" alt="عضو">

          <div class="info">
            <div class="badge" style="background:<?= $statusBg ?>; color:<?= $statusColor ?>">
              <?= $statusText ?>
            </div>

            <div class="row">
              <div class="k">نام و نام خانوادگی:</div>
              <div class="v"><?= $fullName ?></div>
            </div>

            <div class="row">
              <div class="k">کد ملی:</div>
              <div class="v" style="direction:ltr; unicode-bidi:plaintext;">
                  <?= htmlspecialchars($masked) ?>
              </div>
            </div>

            <div class="row">
              <div class="k">وضعیت:</div>
              <div class="v"><?= htmlspecialchars($roleText) ?></div>
            </div>

            <div class="<?= $noteClass ?>">
              <?= htmlspecialchars($officialNoteText) ?>
            </div>

          </div>
        </div>
      </div>

      <div class="footer">
        <div class="note">
          این صفحه صرفاً جهت استعلام اصالت کارت عضویت صادر شده است. هرگونه جعل، تغییر یا سوءاستفاده پیگرد قانونی دارد.
        </div>
        <div class="site">www.ostandar.com</div>
      </div>
    </div>
  </div>
</body>
</html>
